/// <reference path="ScatterPlotVisualisation.ts" />
/// <reference path="../FDEventHandler.ts" />
/// <reference path="../resources/three.d.ts" />
/// <reference path="../debugPrinter.ts" />

/*
 * Created by Diab, Dokter, Hatbauer, Rocnik
 *
 * ivis Group 1 - Summer Term 2013
 *
**/
module FD {
    export class ScatterPlotEventHandler extends FDEventHandler {

        private selectedMesh: THREE.Mesh;
        private mouseDownPosition: THREE.Vector2;
        private visualisation: ScatterPlotVisualisation;
        private infoWindow: string;

        /**
         * Creates a ScatterPlotVisualisationEventHandler object
         *
         * @constructor
         * @param debug: switch debug mode on/off
         * @param fluidDiagram: FluidDiagrams object to connect to the
         *                      scatter plot visualisation
         */
        constructor(debug: boolean, fluidDiagram: FluidDiagrams) {
            super(debug, fluidDiagram);
            this.visualisation = <ScatterPlotVisualisation>(this.fluidDiagrams.getVisualisation())
  }

        /**
         * Register a html UI element
         *
         * @function registerInfoWindow
         * @param container: UI element
         */
        public registerInfoWindow(containerName: string) {
            this.infoWindow = (containerName);
        }

        /**
         * Perform action after a click event
         * Searches for the clicked node and prints
         * the retrieved data to the infoWindow
         *
         * @function onMouseClick
         * @param e: event info
         * @param element: clicked element
         */
        public onMouseClick(e: Event, element) {
            if (element != undefined) {
                if (element instanceof THREE.Mesh) {
                    element = <THREE.Mesh>element;

                    var cNames = this.visualisation.nodes[0].getData();
                    var j = 0;
                    var colNames = [];
                    for (var c in cNames)
                        colNames[j++] = c;

                    for (var node in this.fluidDiagrams.rootNodes) {
                        var currentNode = <FDNode>this.fluidDiagrams.rootNodes[node];
                        if (currentNode.getUniqueId() == element.name.substr(5)) {

                            this.initInfoWindow(currentNode);

                            debugPrinter.print("------------------------------------", this.debug);
                            debugPrinter.print("nodeName: " + element.name.substr(5), this.debug);
                            var data = currentNode.getData();
                            var val = 0;
                            j = 0;
                            for (var d in data) {
                                var colName = colNames[j++];
                                val = parseFloat(data[d]);
                                debugPrinter.print(colName + ": " + val, this.debug);
                            }
                        }
                    }
                }
            }
        }

        /**
         * Record mouse position on move
         *
         * @function onMouseMove
         * @param e: event info
         */
        public onMouseMove(e: Event) {
            if (this.mouseDown == true && this.selectedMesh != null) {
                var x = (<any>e).clientX - this.mouseDownPosition.x;
                var y = (<any>e).clientY - this.mouseDownPosition.y;
                this.mouseDownPosition.x = (<any>e).clientX;
                this.mouseDownPosition.y = (<any>e).clientY;
            }
        }

        /**
         * Respond to user input on the data/icon/color/size 
         * selection and invoke the updating of the plot
         *
         * @function onUserSelection   
         */
        public onUserSelection() {
            debugPrinter.print("User Selection!", this.debug);
            for (var node in this.fluidDiagrams.rootNodes) {
                var n = <FDNode>this.fluidDiagrams.rootNodes[node];
                //   debugPrinter.print("nodeName: " + n.getUniqueId(),this.debug);
                var geom = n.getAllGeometry();
                n.getNodeSceneGraph().remove(geom[0]);
                geom.splice(0, 1);
            }
            this.visualisation.update();
        }

        /**
         * Initialises the infoWindow where the data
         * of a clicked node will be displayed
         *
         * @function initWindow
         * @param clickedElem: the clicked element of which
         *                     the data is to be displayed
         */
        private initInfoWindow(clickedElem: any) {
            var container = <any>document.getElementById(this.infoWindow);
            if (container)
                container.innerHTML = null;
            var table = <any>document.createElement("table");
            var tr = <any>document.createElement("tr");
            var td1 = <any>document.createElement("td");
            var td2 = <any>document.createElement("td");
            var nodeName = <any>document.createElement("nodeName");
            var element = <any>document.createElement("element");
            element.textContent = "element name";
            nodeName.textContent = clickedElem.getUniqueId();
            td1.appendChild(element);
            td2.appendChild(nodeName);
            tr.appendChild(td1);
            tr.appendChild(td2);
            table.appendChild(tr);

            var cNames = this.visualisation.nodes[0].getData();
            var j = 0;
            var colNames = [];
            for (var c in cNames)
                colNames[j++] = c;

            var data = clickedElem.getData();
            var i = 0;

            for (var d in data) {
                var tr = <any>document.createElement("tr");
                var td1 = <any>document.createElement("td");
                var td2 = <any>document.createElement("td");

                var label = <any>document.createElement("label");
                label.textContent = colNames[i++];
                td1.appendChild(label);

                var value = <any>document.createElement("value");
                value.textContent = parseFloat(data[d]);
                td2.appendChild(value);

                tr.appendChild(td1);
                tr.appendChild(td2);
                table.appendChild(tr);
            }
            container.appendChild(table);
        }

        /**
         * Update scenegraph for animation
         * effects etc (not used)
         *
         * @function update
         */
        public update() {
            //animation update here

        }

    }
}